/* Firmware file reading and download helpers
 *
 * See copyright notice in main.c
 */
#include <linux/kernel.h>
#include <linux/firmware.h>

#include "hermes.h"
#include "hermes_dld.h"
#include "orinoco.h"

#include "fw.h"

/* End markers (for Symbol firmware only) */
#define TEXT_END	0x1A		/* End of text header */

struct fw_info {
	char *pri_fw;
	char *sta_fw;
	char *ap_fw;
	u32 pda_addr;
	u16 pda_size;
};

static const struct fw_info orinoco_fw[] = {
	{ NULL, "agere_sta_fw.bin", "agere_ap_fw.bin", 0x00390000, 1000 },
	{ NULL, "prism_sta_fw.bin", "prism_ap_fw.bin", 0, 1024 },
	{ "symbol_sp24t_prim_fw", "symbol_sp24t_sec_fw", NULL, 0x00003100, 512 }
};

/* Structure used to access fields in FW
 * Make sure LE decoding macros are used
 */
struct orinoco_fw_header {
	char hdr_vers[6];       /* ASCII string for header version */
	__le16 headersize;      /* Total length of header */
	__le32 entry_point;     /* NIC entry point */
	__le32 blocks;          /* Number of blocks to program */
	__le32 block_offset;    /* Offset of block data from eof header */
	__le32 pdr_offset;      /* Offset to PDR data from eof header */
	__le32 pri_offset;      /* Offset to primary plug data */
	__le32 compat_offset;   /* Offset to compatibility data*/
	char signature[0];      /* FW signature length headersize-20 */
} __attribute__ ((packed));

/* Download either STA or AP firmware into the card. */
static int
orinoco_dl_firmware(struct orinoco_private *priv,
		    const struct fw_info *fw,
		    int ap)
{
	/* Plug Data Area (PDA) */
	__le16 *pda;

	hermes_t *hw = &priv->hw;
	const struct firmware *fw_entry;
	const struct orinoco_fw_header *hdr;
	const unsigned char *first_block;
	const unsigned char *end;
	const char *firmware;
	struct net_device *dev = priv->ndev;
	int err = 0;

	pda = kzalloc(fw->pda_size, GFP_KERNEL);
	if (!pda)
		return -ENOMEM;

	if (ap)
		firmware = fw->ap_fw;
	else
		firmware = fw->sta_fw;

	printk(KERN_DEBUG "%s: Attempting to download firmware %s\n",
	       dev->name, firmware);

	/* Read current plug data */
	err = hermes_read_pda(hw, pda, fw->pda_addr, fw->pda_size, 0);
	printk(KERN_DEBUG "%s: Read PDA returned %d\n", dev->name, err);
	if (err)
		goto free;

	if (!priv->cached_fw) {
		err = request_firmware(&fw_entry, firmware, priv->dev);

		if (err) {
			printk(KERN_ERR "%s: Cannot find firmware %s\n",
			       dev->name, firmware);
			err = -ENOENT;
			goto free;
		}
	} else
		fw_entry = priv->cached_fw;

	hdr = (const struct orinoco_fw_header *) fw_entry->data;

	/* Enable aux port to allow programming */
	err = hermesi_program_init(hw, le32_to_cpu(hdr->entry_point));
	printk(KERN_DEBUG "%s: Program init returned %d\n", dev->name, err);
	if (err != 0)
		goto abort;

	/* Program data */
	first_block = (fw_entry->data +
		       le16_to_cpu(hdr->headersize) +
		       le32_to_cpu(hdr->block_offset));
	end = fw_entry->data + fw_entry->size;

	err = hermes_program(hw, first_block, end);
	printk(KERN_DEBUG "%s: Program returned %d\n", dev->name, err);
	if (err != 0)
		goto abort;

	/* Update production data */
	first_block = (fw_entry->data +
		       le16_to_cpu(hdr->headersize) +
		       le32_to_cpu(hdr->pdr_offset));

	err = hermes_apply_pda_with_defaults(hw, first_block, pda);
	printk(KERN_DEBUG "%s: Apply PDA returned %d\n", dev->name, err);
	if (err)
		goto abort;

	/* Tell card we've finished */
	err = hermesi_program_end(hw);
	printk(KERN_DEBUG "%s: Program end returned %d\n", dev->name, err);
	if (err != 0)
		goto abort;

	/* Check if we're running */
	printk(KERN_DEBUG "%s: hermes_present returned %d\n",
	       dev->name, hermes_present(hw));

abort:
	/* If we requested the firmware, release it. */
	if (!priv->cached_fw)
		release_firmware(fw_entry);

free:
	kfree(pda);
	return err;
}

/*
 * Process a firmware image - stop the card, load the firmware, reset
 * the card and make sure it responds.  For the secondary firmware take
 * care of the PDA - read it and then write it on top of the firmware.
 */
static int
symbol_dl_image(struct orinoco_private *priv, const struct fw_info *fw,
		const unsigned char *image, const unsigned char *end,
		int secondary)
{
	hermes_t *hw = &priv->hw;
	int ret = 0;
	const unsigned char *ptr;
	const unsigned char *first_block;

	/* Plug Data Area (PDA) */
	__le16 *pda = NULL;

	/* Binary block begins after the 0x1A marker */
	ptr = image;
	while (*ptr++ != TEXT_END);
	first_block = ptr;

	/* Read the PDA from EEPROM */
	if (secondary) {
		pda = kzalloc(fw->pda_size, GFP_KERNEL);
		if (!pda)
			return -ENOMEM;

		ret = hermes_read_pda(hw, pda, fw->pda_addr, fw->pda_size, 1);
		if (ret)
			goto free;
	}

	/* Stop the firmware, so that it can be safely rewritten */
	if (priv->stop_fw) {
		ret = priv->stop_fw(priv, 1);
		if (ret)
			goto free;
	}

	/* Program the adapter with new firmware */
	ret = hermes_program(hw, first_block, end);
	if (ret)
		goto free;

	/* Write the PDA to the adapter */
	if (secondary) {
		size_t len = hermes_blocks_length(first_block);
		ptr = first_block + len;
		ret = hermes_apply_pda(hw, ptr, pda);
		kfree(pda);
		if (ret)
			return ret;
	}

	/* Run the firmware */
	if (priv->stop_fw) {
		ret = priv->stop_fw(priv, 0);
		if (ret)
			return ret;
	}

	/* Reset hermes chip and make sure it responds */
	ret = hermes_init(hw);

	/* hermes_reset() should return 0 with the secondary firmware */
	if (secondary && ret != 0)
		return -ENODEV;

	/* And this should work with any firmware */
	if (!hermes_present(hw))
		return -ENODEV;

	return 0;

free:
	kfree(pda);
	return ret;
}


/*
 * Download the firmware into the card, this also does a PCMCIA soft
 * reset on the card, to make sure it's in a sane state.
 */
static int
symbol_dl_firmware(struct orinoco_private *priv,
		   const struct fw_info *fw)
{
	struct net_device *dev = priv->ndev;
	int ret;
	const struct firmware *fw_entry;

	if (!priv->cached_pri_fw) {
		if (request_firmware(&fw_entry, fw->pri_fw, priv->dev) != 0) {
			printk(KERN_ERR "%s: Cannot find firmware: %s\n",
			       dev->name, fw->pri_fw);
			return -ENOENT;
		}
	} else
		fw_entry = priv->cached_pri_fw;

	/* Load primary firmware */
	ret = symbol_dl_image(priv, fw, fw_entry->data,
			      fw_entry->data + fw_entry->size, 0);

	if (!priv->cached_pri_fw)
		release_firmware(fw_entry);
	if (ret) {
		printk(KERN_ERR "%s: Primary firmware download failed\n",
		       dev->name);
		return ret;
	}

	if (!priv->cached_fw) {
		if (request_firmware(&fw_entry, fw->sta_fw, priv->dev) != 0) {
			printk(KERN_ERR "%s: Cannot find firmware: %s\n",
			       dev->name, fw->sta_fw);
			return -ENOENT;
		}
	} else
		fw_entry = priv->cached_fw;

	/* Load secondary firmware */
	ret = symbol_dl_image(priv, fw, fw_entry->data,
			      fw_entry->data + fw_entry->size, 1);
	if (!priv->cached_fw)
		release_firmware(fw_entry);
	if (ret) {
		printk(KERN_ERR "%s: Secondary firmware download failed\n",
		       dev->name);
	}

	return ret;
}

int orinoco_download(struct orinoco_private *priv)
{
	int err = 0;
	/* Reload firmware */
	switch (priv->firmware_type) {
	case FIRMWARE_TYPE_AGERE:
		/* case FIRMWARE_TYPE_INTERSIL: */
		err = orinoco_dl_firmware(priv,
					  &orinoco_fw[priv->firmware_type], 0);
		break;

	case FIRMWARE_TYPE_SYMBOL:
		err = symbol_dl_firmware(priv,
					 &orinoco_fw[priv->firmware_type]);
		break;
	case FIRMWARE_TYPE_INTERSIL:
		break;
	}
	/* TODO: if we fail we probably need to reinitialise
	 * the driver */

	return err;
}

void orinoco_cache_fw(struct orinoco_private *priv, int ap)
{
#if defined(CONFIG_HERMES_CACHE_FW_ON_INIT) || defined(CONFIG_PM_SLEEP)
	const struct firmware *fw_entry = NULL;
	const char *pri_fw;
	const char *fw;

	pri_fw = orinoco_fw[priv->firmware_type].pri_fw;
	if (ap)
		fw = orinoco_fw[priv->firmware_type].ap_fw;
	else
		fw = orinoco_fw[priv->firmware_type].sta_fw;

	if (pri_fw) {
		if (request_firmware(&fw_entry, pri_fw, priv->dev) == 0)
			priv->cached_pri_fw = fw_entry;
	}

	if (fw) {
		if (request_firmware(&fw_entry, fw, priv->dev) == 0)
			priv->cached_fw = fw_entry;
	}
#endif
}

void orinoco_uncache_fw(struct orinoco_private *priv)
{
#if defined(CONFIG_HERMES_CACHE_FW_ON_INIT) || defined(CONFIG_PM_SLEEP)
	if (priv->cached_pri_fw)
		release_firmware(priv->cached_pri_fw);
	if (priv->cached_fw)
		release_firmware(priv->cached_fw);

	priv->cached_pri_fw = NULL;
	priv->cached_fw = NULL;
#endif
}
