#ifndef __ASM_POWERPC_CPUTABLE_H
#define __ASM_POWERPC_CPUTABLE_H

#include <asm/asm-compat.h>

#define PPC_FEATURE_32			0x80000000
#define PPC_FEATURE_64			0x40000000
#define PPC_FEATURE_601_INSTR		0x20000000
#define PPC_FEATURE_HAS_ALTIVEC		0x10000000
#define PPC_FEATURE_HAS_FPU		0x08000000
#define PPC_FEATURE_HAS_MMU		0x04000000
#define PPC_FEATURE_HAS_4xxMAC		0x02000000
#define PPC_FEATURE_UNIFIED_CACHE	0x01000000
#define PPC_FEATURE_HAS_SPE		0x00800000
#define PPC_FEATURE_HAS_EFP_SINGLE	0x00400000
#define PPC_FEATURE_HAS_EFP_DOUBLE	0x00200000
#define PPC_FEATURE_NO_TB		0x00100000
#define PPC_FEATURE_POWER4		0x00080000
#define PPC_FEATURE_POWER5		0x00040000
#define PPC_FEATURE_POWER5_PLUS		0x00020000
#define PPC_FEATURE_CELL		0x00010000

#ifdef __KERNEL__
#ifndef __ASSEMBLY__

/* This structure can grow, it's real size is used by head.S code
 * via the mkdefs mechanism.
 */
struct cpu_spec;
struct op_powerpc_model;

typedef	void (*cpu_setup_t)(unsigned long offset, struct cpu_spec* spec);

struct cpu_spec {
	/* CPU is matched via (PVR & pvr_mask) == pvr_value */
	unsigned int	pvr_mask;
	unsigned int	pvr_value;

	char		*cpu_name;
	unsigned long	cpu_features;		/* Kernel features */
	unsigned int	cpu_user_features;	/* Userland features */

	/* cache line sizes */
	unsigned int	icache_bsize;
	unsigned int	dcache_bsize;

	/* number of performance monitor counters */
	unsigned int	num_pmcs;

	/* this is called to initialize various CPU bits like L1 cache,
	 * BHT, SPD, etc... from head.S before branching to identify_machine
	 */
	cpu_setup_t	cpu_setup;

	/* Used by oprofile userspace to select the right counters */
	char		*oprofile_cpu_type;

	/* Processor specific oprofile operations */
	struct op_powerpc_model *oprofile_model;
};

extern struct cpu_spec		*cur_cpu_spec;

extern void identify_cpu(unsigned long offset, unsigned long cpu);
extern void do_cpu_ftr_fixups(unsigned long offset);

#endif /* __ASSEMBLY__ */

/* CPU kernel features */

/* Retain the 32b definitions all use bottom half of word */
#define CPU_FTR_SPLIT_ID_CACHE		ASM_CONST(0x0000000000000001)
#define CPU_FTR_L2CR			ASM_CONST(0x0000000000000002)
#define CPU_FTR_SPEC7450		ASM_CONST(0x0000000000000004)
#define CPU_FTR_ALTIVEC			ASM_CONST(0x0000000000000008)
#define CPU_FTR_TAU			ASM_CONST(0x0000000000000010)
#define CPU_FTR_CAN_DOZE		ASM_CONST(0x0000000000000020)
#define CPU_FTR_USE_TB			ASM_CONST(0x0000000000000040)
#define CPU_FTR_604_PERF_MON		ASM_CONST(0x0000000000000080)
#define CPU_FTR_601			ASM_CONST(0x0000000000000100)
#define CPU_FTR_HPTE_TABLE		ASM_CONST(0x0000000000000200)
#define CPU_FTR_CAN_NAP			ASM_CONST(0x0000000000000400)
#define CPU_FTR_L3CR			ASM_CONST(0x0000000000000800)
#define CPU_FTR_L3_DISABLE_NAP		ASM_CONST(0x0000000000001000)
#define CPU_FTR_NAP_DISABLE_L2_PR	ASM_CONST(0x0000000000002000)
#define CPU_FTR_DUAL_PLL_750FX		ASM_CONST(0x0000000000004000)
#define CPU_FTR_NO_DPM			ASM_CONST(0x0000000000008000)
#define CPU_FTR_HAS_HIGH_BATS		ASM_CONST(0x0000000000010000)
#define CPU_FTR_NEED_COHERENT		ASM_CONST(0x0000000000020000)
#define CPU_FTR_NO_BTIC			ASM_CONST(0x0000000000040000)
#define CPU_FTR_BIG_PHYS		ASM_CONST(0x0000000000080000)
#define CPU_FTR_NODSISRALIGN  		ASM_CONST(0x0000000000100000)

#ifdef __powerpc64__
/* Add the 64b processor unique features in the top half of the word */
#define CPU_FTR_SLB           		ASM_CONST(0x0000000100000000)
#define CPU_FTR_16M_PAGE      		ASM_CONST(0x0000000200000000)
#define CPU_FTR_TLBIEL         		ASM_CONST(0x0000000400000000)
#define CPU_FTR_NOEXECUTE     		ASM_CONST(0x0000000800000000)
#define CPU_FTR_IABR  			ASM_CONST(0x0000002000000000)
#define CPU_FTR_MMCRA  			ASM_CONST(0x0000004000000000)
#define CPU_FTR_CTRL			ASM_CONST(0x0000008000000000)
#define CPU_FTR_SMT  			ASM_CONST(0x0000010000000000)
#define CPU_FTR_COHERENT_ICACHE  	ASM_CONST(0x0000020000000000)
#define CPU_FTR_LOCKLESS_TLBIE		ASM_CONST(0x0000040000000000)
#define CPU_FTR_MMCRA_SIHV		ASM_CONST(0x0000080000000000)
#define CPU_FTR_CI_LARGE_PAGE		ASM_CONST(0x0000100000000000)
#define CPU_FTR_PAUSE_ZERO		ASM_CONST(0x0000200000000000)
#else
/* ensure on 32b processors the flags are available for compiling but
 * don't do anything */
#define CPU_FTR_SLB           		ASM_CONST(0x0)
#define CPU_FTR_16M_PAGE      		ASM_CONST(0x0)
#define CPU_FTR_TLBIEL         		ASM_CONST(0x0)
#define CPU_FTR_NOEXECUTE     		ASM_CONST(0x0)
#define CPU_FTR_IABR  			ASM_CONST(0x0)
#define CPU_FTR_MMCRA  			ASM_CONST(0x0)
#define CPU_FTR_CTRL			ASM_CONST(0x0)
#define CPU_FTR_SMT  			ASM_CONST(0x0)
#define CPU_FTR_COHERENT_ICACHE  	ASM_CONST(0x0)
#define CPU_FTR_LOCKLESS_TLBIE		ASM_CONST(0x0)
#define CPU_FTR_MMCRA_SIHV		ASM_CONST(0x0)
#define CPU_FTR_CI_LARGE_PAGE		ASM_CONST(0x0)
#endif

#ifndef __ASSEMBLY__

#define CPU_FTR_PPCAS_ARCH_V2_BASE (CPU_FTR_SLB | \
					CPU_FTR_TLBIEL | CPU_FTR_NOEXECUTE | \
					CPU_FTR_NODSISRALIGN | CPU_FTR_CTRL)

/* iSeries doesn't support large pages */
#ifdef CONFIG_PPC_ISERIES
#define CPU_FTR_PPCAS_ARCH_V2	(CPU_FTR_PPCAS_ARCH_V2_BASE)
#else
#define CPU_FTR_PPCAS_ARCH_V2	(CPU_FTR_PPCAS_ARCH_V2_BASE | CPU_FTR_16M_PAGE)
#endif /* CONFIG_PPC_ISERIES */

/* We only set the altivec features if the kernel was compiled with altivec
 * support
 */
#ifdef CONFIG_ALTIVEC
#define CPU_FTR_ALTIVEC_COMP	CPU_FTR_ALTIVEC
#define PPC_FEATURE_HAS_ALTIVEC_COMP PPC_FEATURE_HAS_ALTIVEC
#else
#define CPU_FTR_ALTIVEC_COMP	0
#define PPC_FEATURE_HAS_ALTIVEC_COMP    0
#endif

/* We need to mark all pages as being coherent if we're SMP or we
 * have a 74[45]x and an MPC107 host bridge.
 */
#if defined(CONFIG_SMP) || defined(CONFIG_MPC10X_BRIDGE)
#define CPU_FTR_COMMON                  CPU_FTR_NEED_COHERENT
#else
#define CPU_FTR_COMMON                  0
#endif

/* The powersave features NAP & DOZE seems to confuse BDI when
   debugging. So if a BDI is used, disable theses
 */
#ifndef CONFIG_BDI_SWITCH
#define CPU_FTR_MAYBE_CAN_DOZE	CPU_FTR_CAN_DOZE
#define CPU_FTR_MAYBE_CAN_NAP	CPU_FTR_CAN_NAP
#else
#define CPU_FTR_MAYBE_CAN_DOZE	0
#define CPU_FTR_MAYBE_CAN_NAP	0
#endif

#define CLASSIC_PPC (!defined(CONFIG_8xx) && !defined(CONFIG_4xx) && \
		     !defined(CONFIG_POWER3) && !defined(CONFIG_POWER4) && \
		     !defined(CONFIG_BOOKE))

enum {
	CPU_FTRS_PPC601 = CPU_FTR_COMMON | CPU_FTR_601 | CPU_FTR_HPTE_TABLE,
	CPU_FTRS_603 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP,
	CPU_FTRS_604 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB | CPU_FTR_604_PERF_MON | CPU_FTR_HPTE_TABLE,
	CPU_FTRS_740_NOTAU = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP,
	CPU_FTRS_740 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_TAU | CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP,
	CPU_FTRS_750 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_TAU | CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP,
	CPU_FTRS_750FX1 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_TAU | CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP |
	    CPU_FTR_DUAL_PLL_750FX | CPU_FTR_NO_DPM,
	CPU_FTRS_750FX2 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_TAU | CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP |
	    CPU_FTR_NO_DPM,
	CPU_FTRS_750FX = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_TAU | CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP |
	    CPU_FTR_DUAL_PLL_750FX | CPU_FTR_HAS_HIGH_BATS,
	CPU_FTRS_750GX = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_MAYBE_CAN_DOZE |
	    CPU_FTR_USE_TB | CPU_FTR_L2CR | CPU_FTR_TAU |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_MAYBE_CAN_NAP |
	    CPU_FTR_DUAL_PLL_750FX | CPU_FTR_HAS_HIGH_BATS,
	CPU_FTRS_7400_NOTAU = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_ALTIVEC_COMP | CPU_FTR_HPTE_TABLE |
	    CPU_FTR_MAYBE_CAN_NAP,
	CPU_FTRS_7400 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB | CPU_FTR_L2CR |
	    CPU_FTR_TAU | CPU_FTR_ALTIVEC_COMP | CPU_FTR_HPTE_TABLE |
	    CPU_FTR_MAYBE_CAN_NAP,
	CPU_FTRS_7450_20 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NEED_COHERENT,
	CPU_FTRS_7450_21 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_L3_DISABLE_NAP |
	    CPU_FTR_NEED_COHERENT,
	CPU_FTRS_7450_23 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_NEED_COHERENT,
	CPU_FTRS_7455_1 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP | CPU_FTR_L3CR |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 | CPU_FTR_HAS_HIGH_BATS |
	    CPU_FTR_NEED_COHERENT,
	CPU_FTRS_7455_20 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_L3_DISABLE_NAP |
	    CPU_FTR_NEED_COHERENT | CPU_FTR_HAS_HIGH_BATS,
	CPU_FTRS_7455 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_HAS_HIGH_BATS |
	    CPU_FTR_NEED_COHERENT,
	CPU_FTRS_7447_10 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_HAS_HIGH_BATS |
	    CPU_FTR_NEED_COHERENT | CPU_FTR_NO_BTIC,
	CPU_FTRS_7447 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_L3CR | CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_HAS_HIGH_BATS |
	    CPU_FTR_NEED_COHERENT,
	CPU_FTRS_7447A = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_L2CR | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_SPEC7450 |
	    CPU_FTR_NAP_DISABLE_L2_PR | CPU_FTR_HAS_HIGH_BATS |
	    CPU_FTR_NEED_COHERENT,
	CPU_FTRS_82XX = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_MAYBE_CAN_DOZE | CPU_FTR_USE_TB,
	CPU_FTRS_G2_LE = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_MAYBE_CAN_DOZE |
	    CPU_FTR_USE_TB | CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_HAS_HIGH_BATS,
	CPU_FTRS_E300 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_MAYBE_CAN_DOZE |
	    CPU_FTR_USE_TB | CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_HAS_HIGH_BATS,
	CPU_FTRS_CLASSIC32 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB | CPU_FTR_HPTE_TABLE,
	CPU_FTRS_POWER3_32 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB | CPU_FTR_HPTE_TABLE,
	CPU_FTRS_POWER4_32 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB | CPU_FTR_HPTE_TABLE | CPU_FTR_NODSISRALIGN,
	CPU_FTRS_970_32 = CPU_FTR_COMMON | CPU_FTR_SPLIT_ID_CACHE |
	    CPU_FTR_USE_TB | CPU_FTR_HPTE_TABLE | CPU_FTR_ALTIVEC_COMP |
	    CPU_FTR_MAYBE_CAN_NAP | CPU_FTR_NODSISRALIGN,
	CPU_FTRS_8XX = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB,
	CPU_FTRS_40X = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_NODSISRALIGN,
	CPU_FTRS_44X = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_NODSISRALIGN,
	CPU_FTRS_E200 = CPU_FTR_USE_TB | CPU_FTR_NODSISRALIGN,
	CPU_FTRS_E500 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_NODSISRALIGN,
	CPU_FTRS_E500_2 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_BIG_PHYS | CPU_FTR_NODSISRALIGN,
	CPU_FTRS_GENERIC_32 = CPU_FTR_COMMON | CPU_FTR_NODSISRALIGN,
#ifdef __powerpc64__
	CPU_FTRS_POWER3 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_IABR,
	CPU_FTRS_RS64 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_IABR |
	    CPU_FTR_MMCRA | CPU_FTR_CTRL,
	CPU_FTRS_POWER4 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_PPCAS_ARCH_V2 | CPU_FTR_MMCRA,
	CPU_FTRS_PPC970 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_PPCAS_ARCH_V2 |
	    CPU_FTR_ALTIVEC_COMP | CPU_FTR_CAN_NAP | CPU_FTR_MMCRA,
	CPU_FTRS_POWER5 = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_PPCAS_ARCH_V2 |
	    CPU_FTR_MMCRA | CPU_FTR_SMT |
	    CPU_FTR_COHERENT_ICACHE | CPU_FTR_LOCKLESS_TLBIE |
	    CPU_FTR_MMCRA_SIHV,
	CPU_FTRS_CELL = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_PPCAS_ARCH_V2 |
	    CPU_FTR_ALTIVEC_COMP | CPU_FTR_MMCRA | CPU_FTR_SMT |
	    CPU_FTR_CTRL | CPU_FTR_PAUSE_ZERO,
	CPU_FTRS_COMPATIBLE = CPU_FTR_SPLIT_ID_CACHE | CPU_FTR_USE_TB |
	    CPU_FTR_HPTE_TABLE | CPU_FTR_PPCAS_ARCH_V2,
#endif

	CPU_FTRS_POSSIBLE =
#ifdef __powerpc64__
	    CPU_FTRS_POWER3 | CPU_FTRS_RS64 | CPU_FTRS_POWER4 |
	    CPU_FTRS_PPC970 | CPU_FTRS_POWER5 | CPU_FTRS_CELL |
            CPU_FTR_CI_LARGE_PAGE |
#else
#if CLASSIC_PPC
	    CPU_FTRS_PPC601 | CPU_FTRS_603 | CPU_FTRS_604 | CPU_FTRS_740_NOTAU |
	    CPU_FTRS_740 | CPU_FTRS_750 | CPU_FTRS_750FX1 |
	    CPU_FTRS_750FX2 | CPU_FTRS_750FX | CPU_FTRS_750GX |
	    CPU_FTRS_7400_NOTAU | CPU_FTRS_7400 | CPU_FTRS_7450_20 |
	    CPU_FTRS_7450_21 | CPU_FTRS_7450_23 | CPU_FTRS_7455_1 |
	    CPU_FTRS_7455_20 | CPU_FTRS_7455 | CPU_FTRS_7447_10 |
	    CPU_FTRS_7447 | CPU_FTRS_7447A | CPU_FTRS_82XX |
	    CPU_FTRS_G2_LE | CPU_FTRS_E300 | CPU_FTRS_CLASSIC32 |
#else
	    CPU_FTRS_GENERIC_32 |
#endif
#ifdef CONFIG_PPC64BRIDGE
	    CPU_FTRS_POWER3_32 |
#endif
#ifdef CONFIG_POWER4
	    CPU_FTRS_POWER4_32 | CPU_FTRS_970_32 |
#endif
#ifdef CONFIG_8xx
	    CPU_FTRS_8XX |
#endif
#ifdef CONFIG_40x
	    CPU_FTRS_40X |
#endif
#ifdef CONFIG_44x
	    CPU_FTRS_44X |
#endif
#ifdef CONFIG_E200
	    CPU_FTRS_E200 |
#endif
#ifdef CONFIG_E500
	    CPU_FTRS_E500 | CPU_FTRS_E500_2 |
#endif
#endif /* __powerpc64__ */
	    0,

	CPU_FTRS_ALWAYS =
#ifdef __powerpc64__
	    CPU_FTRS_POWER3 & CPU_FTRS_RS64 & CPU_FTRS_POWER4 &
	    CPU_FTRS_PPC970 & CPU_FTRS_POWER5 & CPU_FTRS_CELL &
#else
#if CLASSIC_PPC
	    CPU_FTRS_PPC601 & CPU_FTRS_603 & CPU_FTRS_604 & CPU_FTRS_740_NOTAU &
	    CPU_FTRS_740 & CPU_FTRS_750 & CPU_FTRS_750FX1 &
	    CPU_FTRS_750FX2 & CPU_FTRS_750FX & CPU_FTRS_750GX &
	    CPU_FTRS_7400_NOTAU & CPU_FTRS_7400 & CPU_FTRS_7450_20 &
	    CPU_FTRS_7450_21 & CPU_FTRS_7450_23 & CPU_FTRS_7455_1 &
	    CPU_FTRS_7455_20 & CPU_FTRS_7455 & CPU_FTRS_7447_10 &
	    CPU_FTRS_7447 & CPU_FTRS_7447A & CPU_FTRS_82XX &
	    CPU_FTRS_G2_LE & CPU_FTRS_E300 & CPU_FTRS_CLASSIC32 &
#else
	    CPU_FTRS_GENERIC_32 &
#endif
#ifdef CONFIG_PPC64BRIDGE
	    CPU_FTRS_POWER3_32 &
#endif
#ifdef CONFIG_POWER4
	    CPU_FTRS_POWER4_32 & CPU_FTRS_970_32 &
#endif
#ifdef CONFIG_8xx
	    CPU_FTRS_8XX &
#endif
#ifdef CONFIG_40x
	    CPU_FTRS_40X &
#endif
#ifdef CONFIG_44x
	    CPU_FTRS_44X &
#endif
#ifdef CONFIG_E200
	    CPU_FTRS_E200 &
#endif
#ifdef CONFIG_E500
	    CPU_FTRS_E500 & CPU_FTRS_E500_2 &
#endif
#endif /* __powerpc64__ */
	    CPU_FTRS_POSSIBLE,
};

static inline int cpu_has_feature(unsigned long feature)
{
	return (CPU_FTRS_ALWAYS & feature) ||
	       (CPU_FTRS_POSSIBLE
		& cur_cpu_spec->cpu_features
		& feature);
}

#endif /* !__ASSEMBLY__ */

#ifdef __ASSEMBLY__

#define BEGIN_FTR_SECTION		98:

#ifndef __powerpc64__
#define END_FTR_SECTION(msk, val)		\
99:						\
	.section __ftr_fixup,"a";		\
	.align 2;				\
	.long msk;				\
	.long val;				\
	.long 98b;				\
	.long 99b;				\
	.previous
#else /* __powerpc64__ */
#define END_FTR_SECTION(msk, val)		\
99:						\
	.section __ftr_fixup,"a";		\
	.align 3;				\
	.llong msk;				\
	.llong val;				\
	.llong 98b;				\
	.llong 99b;	 			\
	.previous
#endif /* __powerpc64__ */

#define END_FTR_SECTION_IFSET(msk)	END_FTR_SECTION((msk), (msk))
#define END_FTR_SECTION_IFCLR(msk)	END_FTR_SECTION((msk), 0)
#endif /* __ASSEMBLY__ */

#endif /* __KERNEL__ */
#endif /* __ASM_POWERPC_CPUTABLE_H */
